program lndromat;
uses Windows;


///////////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
/////////////////////////////// Consts \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
///////////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
const
 VerData = 'Lndromat v0.99';

 SaveLNDTitle: pchar = 'Save LND file';
 OpenLNDTitle: pchar = 'Open LND file';
 LNDFilter: packed array[1..30] of char = (
  'B', '&', 'W', ' ', 'l', 'a', 'n', 'd', ' ', 'f', 'i', 'l', 'e', 's', ' ', '(', '*', '.', 'l', 'n', 'd', ')', #0, '*', '.', 'L', 'N', 'D', #0,
  #0);

 OpenAltTitle: pchar = 'Load raw height map';
 OpenBumpTitle: pchar = 'Load raw bump map';
 OpenNoiseTitle: pchar = 'Load raw noise map';
 RawFilter: packed array[1..31] of char = (
  'R', 'a', 'w', ' ', 'i', 'm', 'a', 'g', 'e', ' ', 'f', 'i', 'l', 'e', 's', ' ', '(', '*', '.', 'r', 'a', 'w', ')', #0, '*', '.', 'R', 'A', 'W', #0,
  #0);

 OpenDDSTitle: pchar = 'Open DDS file';
 DDSFilter: packed array[1..25] of char = (
  'D', 'D', 'S', ' ', 'f', 'i', 'l', 'e', 's', ' ', '(', '*', '.', 'd', 'd', 's', ')', #0, '*', '.', 'D', 'D', 'S', #0,
  #0);
 DDS4CC: packed array[1..4] of char = ('D', 'D', 'S', ' ');

 OpenTGATitle: pchar = 'Open TGA file(s)';
 TGAFilter: packed array[1..31] of char = (
  'T', 'G', 'A', ' ', 'i', 'm', 'a', 'g', 'e', ' ', 'f', 'i', 'l', 'e', 's', ' ', '(', '*', '.', 't', 'g', 'a', ')', #0, '*', '.', 'T', 'G', 'A', #0,
  #0);

 CloneTexDataTitle: pchar = 'Import texture data?';
 CloneTexDataMsg: pchar = 'Do you want to copy texture data from an existing LND file?';

 LightZDrop: byte = 16; // 0 = sun at horizon; 255 = sun directly overhead
 MinLight: byte = 48;
 Blurray: array[-1..1,-1..1] of byte = ((1, 8, 1), (8, 16, 8), (1, 8, 1));
 BlurDiv: byte = 52;


///////////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
////////////////////////////// Typedefs \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
///////////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
type
///////////////////////// LND crap \\\\\\\\\\\\\\\\\\\\\\\\\
//TLndFile = imaginary record
// Header: TLndHeader;
// LoResTexture: array[1..Header.NumLoResTextures] of TLndLoResTex;
// Block: array[1..Header.NumBlocks] of TLndBlock;
// Country: array[1..Header.NumCountries] of TLndCountry;
// Material: array[1..Header.NumMaterials] of TLndMaterial;
// NoiseMap: array[0..255,0..255] of byte;
// BumpMap: array[0..255,0..255] of byte;
//end;

 TLndHeader = packed record
  NumBlocks: longint;
  BlockIndex: array[0..31,0..31] of byte; // [Y,X]
  NumMaterials: longint;
  NumCountries: longint;
  BlockSize: longint;    // sizeof(TLndBlock)    =   2520
  MaterialSize: longint; // sizeof(TLndMaterial) = 131074
  CountrySize: longint;  // sizeof(TLndCountry)  =   3076
  NumLoResTextures: longword;
 end;

 TLndLoResTex = packed record
  Texture: pointer; // ignore
  Material: pointer; // ignore
  NumSubTextures: longint;
  ID: longint;
  Size: longint;
  // (Size - 4) bytes of DirectDraw texture crap
 end;

 TLndCell = packed record
  Color: array[1..4] of byte; // or something
  Altitude: byte;
  SaveColor: byte;
  Flags: array[1..2] of byte;
 end;
 TLndBlock = packed record
  Cell: array[0..16,0..16] of TLndCell;
  Index: longint;
  MapY: single; // scripting X coord = BlockX * 160
  MapX: single; // scripting Y coord = BlockY * 160
  BlockY: longint; // X coord in Header.BlockIndex
  BlockX: longint; // Y coord in Header.BlockIndex
// <TLNDInfo>
  Clipped: longbool;
  FrameVisibility: longword;
  Unused: longword;
  UseSmallBump: longbool;
  ForceLoResTex: longbool;
   MeshLOD: longword; // enum
   MeshBlending: longword; // enum
   TextureBlend: longword; // enum
   MeshLODType: longword; // ? not defined
   Fog: longword; // ? not defined
// </TLNDInfo>
  Texture: pointer;
  Material: pointer;
  DrawSomething: longint; // ???
  SpecialMaterialBefore: pointer;
  SpecialMaterialAfter: pointer;
  TransformUVBefore: array[0..2,0..3] of single; // ?
  TransformUVAfter: array[0..2,0..3] of single; // ?
  NextSorting: pointer;
  ValueSorting: single;
  LoResTexture: longword;
  fu_lrs: single; // = (iu_lrs / 256)
  fv_lrs: single; // = (iv_lrs / 256)
  iu_lrs: longint; // X coord of LowResTex sub-image
  iv_lrs: longint; // Y coord of LowResTex sub-image
  SmallTexUpdated: longbool;
 end;

 TLNDMapMaterial = packed record
  Index1: longword;
  Index2: longword;
  Coef1: longword;
 end;
 TLNDCountry = packed record
  TerrainType: longword;
  MapMaterial: array[0..255] of TLNDMapMaterial
 end;

 TLNDMaterial = packed record
  TerrainType: word;
  Image: array[0..255,0..255] of word;
 end;

///////////////////////// TGA crap \\\\\\\\\\\\\\\\\\\\\\\\\
//TTargaFile = imaginary record
// Header: TTargaHeader;
// ID: array[Header.IDSize] of byte;
// Palette: array[Header.PaletteLength, (Header.PaletteBits / 8)] of byte;
// ImageDate: array of byte;
//end;

 TTargaHeader = packed record
  IDSize: byte;
  UsePalette: byte; // 0
  ImageType: byte; // 2
  PaletteOrigin: word;
  PaletteLength: word;
  PaletteBits: byte;
  XOrigin: word;
  YOrigin: word;
  Width: word; // 256
  Height: word; // 256
  BPP: byte; // 16
  Properties: byte; // 32?
 end;

///////////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
//////////////////////////////// Vars \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
///////////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
var
 LND, DataFile: HANDLE;
 LNDOFN, DataOFN: OPENFILENAME;
 FNbuf, FNbuf2: array[1..320] of char;
 MatBuf: array[1..512] of char;
 MatDir, MatFile: pchar;
 Header, Header2: TLndHeader;
 LoResTexture: TLndLoResTex;
 Block: array[1..255] of TLndBlock;
 Country: TLNDCountry;
 Material: TLNDMaterial;
 NoiseMap, BumpMap: array[0..255,0..255] of byte;
 BigMap, LightMap: array[0..511,0..511] of byte;
 Magic4CC: packed array[1..4] of char;
 CloningTexData: boolean;
 TgaHeader: TTargaHeader;
 P: pointer;
 X, Y, Z, BytesRead, BytesWritten: longword;
 i, j: word;
 m, n: integer;
 LightZ, OldZ, ThisZ, ThisLight, MaxX, MaxY: byte;
 LightFactor: real;



///////////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
//////////////////////////////// Code \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
///////////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
begin
 writeln(VerData);
 writeln;

 DataOFN.lStructSize := sizeof(DataOFN);
 DataOFN.lpstrFile := @FNbuf;
 DataOFN.nMaxFile := sizeof(FNbuf);
 DataOFN.Flags := OFN_EXPLORER + OFN_FILEMUSTEXIST + OFN_HIDEREADONLY;

 //////////////// get height map \\\\\\\\\\\\\\\\

 FNbuf := 'HeightMap.raw';
 DataOFN.lpstrFilter := @RawFilter;
 DataOFN.lpstrTitle := OpenAltTitle;
 if GetOpenFileName(@DataOFN) then
  begin
   writeln('Loading ', FNbuf, '...');
   DataFile := CreateFile(@FNbuf, GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
   if (DataFile = INVALID_HANDLE_VALUE) then
    writeln(' Error ', GetLastError, ' opening height map!')
   else
    begin
     ReadFile(DataFile, BigMap, SizeOf(BigMap), BytesRead, nil);
     CloseHandle(DataFile);
    end;
  end;

 ///////////////// get block index \\\\\\\\\\\\\\\\

 Header.NumBlocks := 0;

 for Y := 0 to 31 do
  for X := 0 to 31 do
   for i := 0 to 15 do
    for j := 0 to 15 do
     Header.BlockIndex[Y,X] := Header.BlockIndex[Y,X] or BigMap[(Y * 16 + i), (X * 16 + j)];

 // match up index and blocks
 for Y := 0 to 31 do
  for X := 0 to 31 do
   if (Header.BlockIndex[Y,X] > 0) then
    begin
     inc(Header.NumBlocks, 1);

     if (Header.NumBlocks > 255) then
      begin
       writeln('Too many blocks in index (max is 255)!');
       exit;
      end;

     Block[Header.NumBlocks].MapY := Y * 160;
     Block[Header.NumBlocks].MapX := X * 160;
     Block[Header.NumBlocks].BlockY := Y;
     Block[Header.NumBlocks].BlockX := X;

     Header.BlockIndex[Y,X] := Header.NumBlocks;
    end;

 if (Header.NumBlocks = 0) then
  begin
   writeln('No blocks in index!');
   exit;
  end;

 Header.NumLoResTextures := Header.NumBlocks shr 4; // one LoResTex covers 4x4 = 16 blocks -- right?
 if ((Header.NumBlocks and 15) > 0) then
  inc(Header.NumLoResTextures, 1);

 ///////////// calculate light levels \\\\\\\\\\\\\

 for X := 0 to 511 do
  begin // top right
   LightZ := 0;
   OldZ := 0;

   for Y := 0 to (511 - X) do
    begin // cross map diagonally
     ThisZ := BigMap[Y, (X + Y)];


     // TODO: figure out math for how split affects this
     LightFactor := (LightZDrop + ThisZ - LightZ) / (LightZDrop * 2);
     if LightFactor > 1 then
      LightFactor := 1
     else if LightFactor < 0 then
      LightFactor := 0;

     ThisLight := MinLight + trunc(LightFactor * (255 - MinLight));


     if (ThisZ < (LightZ - LightZDrop)) then
      if LightZ > LightZDrop then
       dec(LightZ, LightZDrop)
      else
       LightZ := 0
     else
      LightZ := ThisZ;

     LightMap[Y, (X + Y)] := ThisLight;
     OldZ := ThisZ;
    end;
  end;

 for Y := 1 to 511 do
  begin // lower left
   LightZ := 0;
   OldZ := 0;

   for X := 0 to (511 - Y) do
    begin // cross map diagonally
     ThisZ := BigMap[(Y + X), X];


     // TODO: figure out math for how split affects this
     LightFactor := (LightZDrop + ThisZ - LightZ) / (LightZDrop * 2);
     if LightFactor > 1 then
      LightFactor := 1
     else if LightFactor < 0 then
      LightFactor := 0;

     ThisLight := MinLight + trunc(LightFactor * (255 - MinLight));


     if (ThisZ < (LightZ - LightZDrop)) then
      if LightZ > LightZDrop then
       dec(LightZ, LightZDrop)
      else
       LightZ := 0
     else
      LightZ := ThisZ;

     LightMap[(Y + X), X] := ThisLight;
     OldZ := ThisZ;
    end;
  end;


 ///////////////// fill in blocks \\\\\\\\\\\\\\\\\

 for i := 1 to Header.NumBlocks do
  begin
   Block[i].Index := i; // right?

   if (Block[i].BlockX = 31) then
    MaxX := 15
   else
    MaxX := 16;

   if (Block[i].BlockY = 31) then
    MaxY := 15
   else
    MaxY := 16;

   // vertex stuff
   for Y := 0 to MaxY do
    for X := 0 to MaxX do
     begin
       Block[i].Cell[Y,X].Altitude := BigMap[(Block[i].BlockY * 16 + Y), (Block[i].BlockX * 16 + X)];

       if ((Block[i].BlockY * 16 + Y) in [1..510]) and ((Block[i].BlockX * 16 + X) in [1..510]) then // blur
        begin
         Z := 0;

         for n := -1 to 1 do
          for m := -1 to 1 do
           inc(Z, (LightMap[(Block[i].BlockY * 16 + Y + n), (Block[i].BlockX * 16 + X + m)] * Blurray[n, m]));

         ThisLight := Z DIV BlurDiv;
        end
       else // no blur (too close to edge)
        ThisLight := LightMap[(Block[i].BlockY * 16 + Y), (Block[i].BlockX * 16 + X)];

       Block[i].Cell[Y,X].SaveColor := ThisLight;
       Block[i].Cell[Y,X].Color[4] := ThisLight;
     end;

   // cell stuff
   for Y := 0 to 15 do
    for X := 0 to 15 do
     if (Block[i].Cell[Y,X].Altitude < 2) then // water
      Block[i].Cell[Y,X].Flags[1] := 16
     else if (Block[i].Cell[Y,X].Altitude = 2) then // coast
      Block[i].Cell[Y,X].Flags[1] := 32;

   Block[i].LoResTexture := ((i - 1) shr 4); // right?

   Block[i].iu_lrs := ((i - 1) and 3) * 64; // X coord of LowResTex sub-image
   Block[i].iv_lrs := (((i - 1) shr 2) and 3) * 64; // Y coord of LowResTex sub-image

   Block[i].fu_lrs := (Block[i].iu_lrs / 256);
   Block[i].fv_lrs := (Block[i].iv_lrs / 256);

  end;
 writeln;


//////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
////////////////////// get image data \\\\\\\\\\\\\\\\\\\\\\
//////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
 DataOFN.Flags := OFN_EXPLORER + OFN_FILEMUSTEXIST;
 /////////// low-res textures \\\\\\\\\\\

 FNbuf := '';
 DataOFN.lpstrFilter := @DDSFilter;
 DataOFN.lpstrTitle := OpenDDSTitle;
 if GetOpenFileName(@DataOFN) then
  begin
   writeln('Loading ', FNbuf, '...');
   DataFile := CreateFile(@FNbuf, GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
   if (DataFile = INVALID_HANDLE_VALUE) then
     writeln('Error ', GetLastError, ' opening DDS file!')
   else
    begin
     LoResTexture.Size := GetFileSize(DataFile, nil);
     ReadFile(DataFile, Magic4CC, SizeOf(Magic4CC), BytesRead, nil);
     if (AnsiString(Magic4CC) = AnsiString(DDS4CC)) then
      begin
       P := GetMem(LoResTexture.Size - 4);

       ReadFile(DataFile, P^, (LoResTexture.Size - 4), BytesRead, nil);
      end
     else
      LoResTexture.Size := 4;

     CloseHandle(DataFile);
    end;
   writeln;
  end;

 //////////// get user input \\\\\\\\\\\\

 if (MessageBox(0, CloneTexDataMsg, CloneTexDataTitle, MB_YESNO) = IDYES) then
  begin
   DataOFN.lpstrFilter := @LNDFilter;
   DataOFN.lpstrTitle := OpenLNDTitle;

   if GetOpenFileName(@DataOFN) then
    begin
     writeln('Loading ', FNbuf, '...');
     DataFile := CreateFile(@FNbuf, GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
     if (DataFile = INVALID_HANDLE_VALUE) then
       writeln('Error ', GetLastError, ' opening LND file!')
     else
      begin
       CloningTexData := true;

       writeln(' Reading header...');
       ReadFile(DataFile, Header2, SizeOf(Header2), BytesRead, nil);

       Header.NumMaterials := Header2.NumMaterials;
       Header.NumCountries := Header2.NumCountries;

       writeln(' Skipping ', Header2.NumLoResTextures ,' low-res textures...');
       for i := 1 to Header2.NumLoResTextures do
        begin // skip low-res textures
         ReadFile(DataFile, LoResTexture, SizeOf(LoResTexture), BytesRead, nil);
         SetFilePointer(DataFile, (LoResTexture.Size - 4), nil, FILE_CURRENT);
        end;

       writeln(' Skipping blocks...');
       SetFilePointer(DataFile, ((Header2.NumBlocks - 1) * Header2.BlockSize), nil, FILE_CURRENT); // skip blocks

      end;

     writeln;
    end;
  end;

 if not CloningTexData then
  begin
   repeat
    write('Desired number of countries (1-16)? '); readln(Header.NumCountries);
    writeln;
   until (Header.NumCountries > 0) and (Header.NumCountries < 17);

   /////////// material textures \\\\\\\\\\

   Header.NumMaterials := 1; // default

   DataOFN.lpstrFile := @MatBuf;
   DataOFN.nMaxFile := sizeof(MatBuf);
   DataOFN.Flags := OFN_EXPLORER or OFN_FILEMUSTEXIST or OFN_ALLOWMULTISELECT;
   DataOFN.lpstrFilter := @TGAFilter;
   DataOFN.lpstrTitle := OpenTGATitle;
   if GetOpenFileName(@DataOFN) then
    begin
     MatDir := @MatBuf;

     if ((GetFileAttributes(MatDir) and FILE_ATTRIBUTE_DIRECTORY) = FILE_ATTRIBUTE_DIRECTORY) then
      begin // count filenames
       writeln('Materials in ', MatDir, ':');

       i := Length(MatDir) + 2;
       Header.NumMaterials := 0;
       MatFile := @MatBuf[i];
       if MatFile <> '' then
        repeat
         writeln(' ', MatFile);
         inc(i, Length(MatFile) + 1);
         inc(Header.NumMaterials, 1);
         MatFile := @MatBuf[i];
        until MatFile = '';

       writeln;
      end;
    end;

   DataOFN.lpstrFile := @FNbuf;
   DataOFN.nMaxFile := sizeof(FNbuf);
   DataOFN.Flags := OFN_EXPLORER + OFN_FILEMUSTEXIST;

   ////////// bump map \\\\\\\\\\

   FNbuf := 'BumpMap.raw';
   DataOFN.lpstrFilter := @RawFilter;
   DataOFN.lpstrTitle := OpenBumpTitle;
   if GetOpenFileName(@DataOFN) then
    begin
     writeln('Loading ', FNbuf, '...');
     DataFile := CreateFile(@FNbuf, GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
     if (DataFile = INVALID_HANDLE_VALUE) then
       writeln(' Error ', GetLastError, ' opening bump map!')
     else
      begin
       ReadFile(DataFile, BumpMap, SizeOf(BumpMap), BytesRead, nil);
       CloseHandle(DataFile);
      end;
     writeln;
    end;

   ////////// noise map \\\\\\\\\

   FNbuf := 'NoiseMap.raw';
   DataOFN.lpstrFilter := @RawFilter;
   DataOFN.lpstrTitle := OpenNoiseTitle;
   if GetOpenFileName(@DataOFN) then
    begin
     writeln('Loading ', FNbuf, '...');
     DataFile := CreateFile(@FNbuf, GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
     if (DataFile = INVALID_HANDLE_VALUE) then
       writeln('Error ', GetLastError, ' opening noise map!')
     else
      begin
       ReadFile(DataFile, NoiseMap, SizeOf(NoiseMap), BytesRead, nil);
       CloseHandle(DataFile);
      end;
     writeln;
    end;
  end;

 Header.BlockSize := 2520;
 Header.MaterialSize := sizeof(Material);
 Header.CountrySize := sizeof(Country);


//////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
////////////////////// write lnd file \\\\\\\\\\\\\\\\\\\\\\
//////////////////////////////\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

 FNbuf2 := 'untitled.lnd';
 LNDOFN.lStructSize := sizeof(DataOFN);
 LNDOFN.lpstrFilter := @LNDFilter;
 LNDOFN.lpstrFile := @FNbuf2;
 LNDOFN.nMaxFile := sizeof(FNbuf2);
 LNDOFN.lpstrTitle := SaveLNDTitle;
 LNDOFN.Flags := OFN_EXPLORER + OFN_OVERWRITEPROMPT;
 if GetSaveFileName(@LNDOFN) then
   writeln('Creating ', FNbuf2, '...')
 else // GetSaveFileName() failed (user probably cancelled).
  exit;

 LND := CreateFile(@FNbuf2, (GENERIC_READ + GENERIC_WRITE), FILE_SHARE_READ, nil, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0);
 if (LND = INVALID_HANDLE_VALUE) then
  begin
   writeln(' Error ', GetLastError, ' opening output file!');
   exit;
  end;

 ///////////////////// header \\\\\\\\\\\\\\\\\\\\\

 writeln(' Writing header...');
 inc(Header.NumBlocks, 1); // why the hell...?
 WriteFile(LND, Header, SizeOf(Header), BytesWritten, nil);
 writeln('  NumBlocks        ', Header.NumBlocks);
 writeln('  NumMaterials     ', Header.NumMaterials);
 writeln('  NumCountries     ', Header.NumCountries);
 writeln('  BlockSize        ', Header.BlockSize);
 writeln('  MaterialSize     ', Header.MaterialSize);
 writeln('  CountrySize      ', Header.CountrySize);
 writeln('  NumLoResTextures ', Header.NumLoResTextures);
 writeln;

 //////////////// low-res textures \\\\\\\\\\\\\\\\

 writeln(' Writing low-res textures...');
 for i := 1 to Header.NumLoResTextures do
  begin // write low-res textures
   LoResTexture.ID := i - 1;
   if (i < Header.NumLoResTextures) then
    LoResTexture.NumSubTextures := 16
   else
    LoResTexture.NumSubTextures := (Header.NumBlocks - 1) and $F;

   WriteFile(LND, LoResTexture, SizeOf(LoResTexture), BytesWritten, nil);
   if (LoResTexture.Size > 4) then
    WriteFile(LND, P^, (LoResTexture.Size - 4), BytesWritten, nil);

  end;

 FreeMem(P, (LoResTexture.Size - 4));

 ///////////////////// blocks \\\\\\\\\\\\\\\\\\\\\

 writeln(' Writing blocks...');
 for i := 2 to Header.NumBlocks do // why the hell...?
  WriteFile(LND, Block[i - 1], Header.BlockSize, BytesWritten, nil);

 ////////////////// texture data \\\\\\\\\\\\\\\\\\

 if not CloningTexData then
  begin
   // countries 
   for i := 0 to 255 do
    Country.MapMaterial[i].Coef1 := 255;

   writeln(' Writing countries...');
   for i := 1 to Header.NumCountries do // write countries
    WriteFile(LND, Country, SizeOf(Country), BytesWritten, nil);

   // materials
   writeln(' Writing materials...');
   if ((GetFileAttributes(MatDir) and FILE_ATTRIBUTE_DIRECTORY) = FILE_ATTRIBUTE_DIRECTORY) then
    begin
      SetCurrentDirectory(MatDir);
      j := Length(MatDir) + 2;
      MatFile := @MatBuf[j];
    end
   else // stupid multi-select
    begin
     MatFile := MatDir;
     j := 1;
    end;

   repeat
    writeln('  Loading ', MatFile, '...');
    DataFile := CreateFile(MatFile, GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
    if (DataFile = INVALID_HANDLE_VALUE) then
     writeln('  Error ', GetLastError, ' opening TGA file!')
    else
     begin
      ReadFile(DataFile, TgaHeader, SizeOf(TgaHeader), BytesRead, nil);
      writeln('   TGA header:');
      writeln('    IDSize: ', TgaHeader.IDSize);
      writeln('    UsePalette: ', TgaHeader.UsePalette);
      writeln('    ImageType: ', TgaHeader.ImageType);
      writeln('    PaletteOrigin: ', TgaHeader.PaletteOrigin);
      writeln('    PaletteLength: ', TgaHeader.PaletteLength);
      writeln('    PaletteBits: ', TgaHeader.PaletteBits);
      writeln('    XOrigin: ', TgaHeader.XOrigin);
      writeln('    YOrigin: ', TgaHeader.YOrigin);
      writeln('    Width: ', TgaHeader.Width);
      writeln('    Height: ', TgaHeader.Height);
      writeln('    BPP: ', TgaHeader.BPP);
      writeln('    Properties: ', TgaHeader.Properties);

      if (TgaHeader.ImageType <> 2) or ((TgaHeader.Properties and $DE) <> 0) then
       writeln('  Wrong image type. Material textures must be 16bpp, uncompressed, and non-interlaced.')
      else if (TgaHeader.Width <> 256) or (TgaHeader.Height <> 256) then
       writeln('  Wrong image size. Material textures must be 256x256 pixels.')
      else if (TgaHeader.BPP <> 16) then
       writeln('  Wrong bit depth. Material textures must have 16 bits per pixel.')
      else
       begin
        SetFilePointer(DataFile, TgaHeader.IDSize, nil, FILE_CURRENT);
        ReadFile(DataFile, Material.Image, SizeOf(Material.Image), BytesRead, nil);
       end;

      CloseHandle(DataFile);
     end;

    WriteFile(LND, Material, SizeOf(Material), BytesWritten, nil);
    writeln;

    inc(j, Length(MatFile) + 1);
    MatFile := @MatBuf[j];
   until MatFile = '';

  end
 else // CloningTexData
  begin
   writeln(' Writing countries...');
   for i := 1 to Header.NumCountries do
    begin
     ReadFile(DataFile, Country, SizeOf(Country), BytesRead, nil);
     WriteFile(LND, Country, SizeOf(Country), BytesWritten, nil);
    end;

   // materials
   writeln(' Writing materials...');
   for i := 1 to Header.NumMaterials do
    begin
     ReadFile(DataFile, Material, SizeOf(Material), BytesRead, nil);
     WriteFile(LND, Material, SizeOf(Material), BytesWritten, nil);
    end;

   ReadFile(DataFile, NoiseMap, SizeOf(NoiseMap), BytesRead, nil);
   ReadFile(DataFile, BumpMap, SizeOf(BumpMap), BytesRead, nil);

   CloseHandle(DataFile);
  end;

 // bump and noise maps

 writeln(' Writing bump and noise maps...');
 WriteFile(LND, NoiseMap, SizeOf(NoiseMap), BytesWritten, nil);
 WriteFile(LND, BumpMap, SizeOf(BumpMap), BytesWritten, nil);

 ////////////// close file \\\\\\\\\\\\\\

 writeln;
 CloseHandle(LND);
 writeln('Done.');
end.
